const { app, BrowserWindow, ipcMain } = require('electron');
const path = require('path');
const fs = require('fs');
const https = require('https');
const { exec, spawn } = require('child_process');
const semver = require('semver');

class UpdaterCustom {
    constructor(mainWindow, updateUrl) {
        this.mainWindow = mainWindow;
        this.updateUrl = updateUrl; 
        this.pollInterval = 60 * 60 * 1000; // 1 hora
        
        this.currentVersion = app.getVersion();
        this.downloadPercentage = 0;
        this.downloadedFilePath = null;
        this.remoteVersionInfo = null;

        this.setupIPC();
    }

    setupIPC() {
        ipcMain.on('updater:check', () => this.checkForUpdates());
        ipcMain.on('updater:download', () => {
             if (this.remoteVersionInfo) this.downloadUpdate(this.remoteVersionInfo);
        });
        ipcMain.on('updater:install', () => this.installUpdate());
    }

    start() {
        console.log('[Updater] Started. Checking updates...');
        // Verificar al inicio
        setTimeout(() => this.checkForUpdates(), 10000);
        // Y cada hora
        setInterval(() => this.checkForUpdates(), this.pollInterval);
    }

    log(msg) {
        console.log(`[Updater] ${msg}`);
         if (this.mainWindow && !this.mainWindow.isDestroyed()) {
             this.mainWindow.webContents.send('updater:log', msg);
         }
    }

    async checkForUpdates() {
        this.log('Checking for updates...');
        try {
            const versionUrl = `${this.updateUrl}/version.json?t=${new Date().getTime()}`;
            
            // Usando node-fetch (si version node < 18, usar https nativo o axios. Electron v36 tiene fetch nativo)
            const response = await fetch(versionUrl);
            if (!response.ok) throw new Error(`HTTP Error ${response.status}`);
            
            const data = await response.json();
            
            this.log(`Remote: ${data.version}, Current: ${this.currentVersion}`);
            
            if (semver.gt(data.version, this.currentVersion)) {
                this.remoteVersionInfo = data;
                this.mainWindow.webContents.send('updater:available', data);
            } else {
                this.mainWindow.webContents.send('updater:not-available');
            }
            
        } catch (error) {
            this.log(`Error checking updates: ${error.message}`);
            this.mainWindow.webContents.send('updater:error', error.message);
        }
    }

    downloadUpdate(info) {
        const fileUrl = `${this.updateUrl}/${info.url}`;
        const fileName = info.url;
        const downloadPath = path.join(app.getPath('temp'), fileName);
        
        this.log(`Downloading ${fileUrl} to ${downloadPath}`);
        this.mainWindow.webContents.send('updater:download-start');

        const file = fs.createWriteStream(downloadPath);
        
        https.get(fileUrl, (response) => {
            if (response.statusCode !== 200) {
                this.mainWindow.webContents.send('updater:error', `Download failed: ${response.statusCode}`);
                return;
            }

            const totalBytes = parseInt(response.headers['content-length'], 10);
            let downloadedBytes = 0;

            response.on('data', (chunk) => {
                downloadedBytes += chunk.length;
                file.write(chunk);
                
                if (totalBytes) {
                    const percent = (downloadedBytes / totalBytes) * 100;
                    if (percent - this.downloadPercentage > 1 || percent === 100) {
                        this.downloadPercentage = percent;
                        this.mainWindow.webContents.send('updater:download-progress', {
                            percent: percent,
                            transferred: downloadedBytes,
                            total: totalBytes
                        });
                    }
                }
            });

            response.on('end', () => {
                file.end();
                this.log('Download complete');
                this.downloadedFilePath = downloadPath;
                this.mainWindow.webContents.send('updater:downloaded', info);
            });

        }).on('error', (err) => {
            fs.unlink(downloadPath, () => {}); 
            this.log(`Download error: ${err.message}`);
            this.mainWindow.webContents.send('updater:error', err.message);
        });
    }

    installUpdate() {
        if (!this.downloadedFilePath) return;

        this.log('Installing update...');
        
        // En desarrollo __dirname es 'electron/', en prod depende.
        // Asumimos que install-update.sh está junto a este archivo
        const scriptPath = path.join(__dirname, 'install-update.sh');
        
        // Copiar el script a temporal para poder ejecutarlo sin conflictos si la carpeta app se borra
        const tempScriptPath = '/tmp/install-update-running.sh';
        try {
            fs.copyFileSync(scriptPath, tempScriptPath);
            fs.chmodSync(tempScriptPath, '0755');
        } catch (e) {
            this.log(`Error copying script to temp: ${e.message}`);
            // Fallback to original path if copy fails
        }
        
        const finalScriptPath = fs.existsSync(tempScriptPath) ? tempScriptPath : scriptPath;

        // Determine App Dir
        // app.getAppPath() -> .../resources/app (inside package)
        // We want the directory containing the executable and 'resources' folder
        // Logic: path.dirname(app.getPath('exe')) gives the folder with the binary
        const appDir = path.dirname(app.getPath('exe'));
        
        // Use systemd-run to spawn a transient service that survives our restart
        // Requires 'sudo' (user pi has nopasswd sudo confirmed)
        const runId = `kiosk-updater-${Date.now()}`;
        const args = [
        'systemd-run',
        `--unit=${runId}`,
        '--description=Update Kiosk',
        '--collect', // Limpia la unidad al terminar
        '/bin/bash',
        finalScriptPath,
        this.downloadedFilePath
    ];


        const subprocess = spawn('sudo', args, {
        detached: true,   // Crea su propio grupo de procesos
        stdio: 'ignore'   // IMPORTANTE: Corta la comunicación de logs para poder salir
    });

    // 3. Manejo de errores INICIALES (solo si falla al nacer)
    subprocess.on('error', (err) => {
        this.log(`❌ Error al hacer spawn: ${err.message}`);
        this.mainWindow.webContents.send('updater:error', 'Failed to spawn update process');
    });

    // 4. "Desenchufar" el proceso de Node.js
    subprocess.unref();

    // 5. Notificar y salir
    this.log(`✅ Update lanzado (ID: ${runId}). Cerrando aplicación...`);
    this.mainWindow.webContents.send('updater:log', 'Update started. Rebooting...');

    // Le damos un segundo a la UI para mostrar el mensaje y cerramos
    setTimeout(() => {
        app.quit();
    }, 1000);
        
        // Quit App
        setTimeout(() => app.quit(), 1000);
    }
}

module.exports = UpdaterCustom;
