const { app, BrowserWindow, ipcMain } = require('electron');
const url = require("url");
const path = require("path");
const { SerialPort } = require('serialport');
const { ReadlineParser } = require('@serialport/parser-readline');
const Store = require('electron-store');
const os = require('os');
const dev = process.argv.includes('--dev');
const store = new Store();
let mainWindow;
let currentPort = null;
let secondaryPort = null;
let parser = null;
let simulationInterval = null;
let isSimulationMode = false;

// Reconnection State
let isIntentionalDisconnect = false;
let reconnectTimer = null;
let isIntentionalDisconnectSecondary = false;
let reconnectTimerSecondary = null;
let isSimulationModeSecondary = false;
let simulationIntervalSecondary = null;

// Configuration
// const BAUD_RATE = 115200; // Removed constant, now dynamic

function createWindow() {
  mainWindow = new BrowserWindow({
    width: 1024,
    height: 768,
    webPreferences: {
      nodeIntegration: true,
      contextIsolation: false, // For easier prototyping as per plan
    },
    fullscreen: !dev,
    frame: dev, // Quita los bordes de la ventana
  });
  if (dev) {
    mainWindow.loadURL('http://localhost:4200');
  } else {
    mainWindow.loadURL(
      url.format({
        pathname: path.join(__dirname, `../dist/electron-app-raspberry/browser/index.html`),
        protocol: "file:",
        slashes: true
      })
    );
  }
  if(dev){
    mainWindow.webContents.openDevTools();
  }

  mainWindow.on('closed', function () {
    mainWindow = null;
    stopSimulation();
    if (currentPort && currentPort.isOpen) {
      currentPort.close();
    }
    if (secondaryPort && secondaryPort.isOpen) {
      secondaryPort.close();
    }
  });

  // Auto-connect attempt after window loads
  mainWindow.webContents.once('did-finish-load', () => {
    attemptAutoConnect();
    
    // Initialize Auto Updater
    const UpdaterCustom = require('./updater-custom');
    const updater = new UpdaterCustom(mainWindow, 'https://siadven.mx/actualizaciones');
    updater.start();
  });
}

// --- Serial Port Logic ---

async function listPorts() {
  try {
    const ports = await SerialPort.list();
    return ports;
  } catch (err) {
    console.error('Error listing ports:', err);
    return [];
  }
}
async function getIP() {
  try {
    const ip = await os.networkInterfaces();
    return ip;
  } catch (err) {
    console.error('Error getting IP:', err);
    return null;
  }
}

function connectToPort(portPath, baudRate = 115200) {
  // If we are simulating, stop it
  if (isSimulationMode) stopSimulation();

  // Reset intention flag on new connection attempt
  isIntentionalDisconnect = false;
  if (reconnectTimer) {
    clearTimeout(reconnectTimer);
    reconnectTimer = null;
  }

  if (currentPort && currentPort.isOpen) {
    // If we are switching ports, we treat the closure of old port as 'intentional' in a way,
    // or just direct close. But here we usually close before opening new.
    // If we call connectToPort while open, we must close first.
    // To prevent reconnect loop on this specific close, we can set flag temporarily?
    // Actually, usually connectToPort is called by user action, so persistence is fine.
    // But let's safe close:
    currentPort.close();
    // The 'close' event will fire. We need to make sure IT doesn't trigger reconnect if we are just switching.
    // But since we set `isIntentionalDisconnect = false` above, the close handler MIGHT think it's accidental?
    // Wait, if I call close(), it's synchronous-ish. 
    // Let's set a temp flag or just rely on 'isIntentionalDisconnect' being managed by the CALLER usually?
    // The only caller is 'serial:connect' or 'attemptAutoConnect' or 'attemptReconnect'.

    // Simplification: logic below handles the "Close existing" as intentional? 
    // Actually, if I am manually connecting to a NEW port, I likely want the old one closed intentionally.
    // But wait, `isIntentionalDisconnect` is global. 
    // Let's rely on logic: if we are here, we are connecting.
  }

  try {
    currentPort = new SerialPort({ path: portPath, baudRate: parseInt(baudRate) });
    parser = currentPort.pipe(new ReadlineParser({ delimiter: '\n' }));

    currentPort.on('open', () => {
      console.log(`Connected to ${portPath}`);
      if (mainWindow) mainWindow.webContents.send('serial:status', { connected: true, port: portPath });

      // Save for persistence
      savePortInfo(portPath, baudRate);
    });

    currentPort.on('error', (err) => {
      console.error('Serial Port Error:', err);
      // Only send error status if intentional disconnect or if we want to show it.
      // If retrying, we might want to show 'Reconnecting' with error?
      if (!isIntentionalDisconnect) {
        if (mainWindow) mainWindow.webContents.send('serial:status', { connected: false, error: err.message, reconnecting: true });

        if (!currentPort.isOpen) {
          attemptReconnect(portPath, baudRate, false);
        }
      } else {
        if (mainWindow) mainWindow.webContents.send('serial:status', { connected: false, error: err.message });
      }
    });

    currentPort.on('close', () => {
      console.log('Port closed');

      if (!isIntentionalDisconnect) {
        console.log('Port disconnected unexpectedly. Attempting reconnect...');
        // Send reconnecting status immediately
        if (mainWindow) mainWindow.webContents.send('serial:status', { connected: false, reconnecting: true });
        attemptReconnect(portPath, baudRate, false);
      } else {
        console.log('Port disconnected intentionally.');
        if (mainWindow) mainWindow.webContents.send('serial:status', { connected: false });
      }
    });

    parser.on('data', (data) => {
      handleSerialData(data);
    });

    return { success: true };
  } catch (err) {
    return { success: false, error: err.message };
  }
}

function attemptReconnect(portPath, baudRate, isSecondary) {
  const retryDelay = 3000; // 3 seconds

  if (isSecondary) {
    if (reconnectTimerSecondary) clearTimeout(reconnectTimerSecondary);
    reconnectTimerSecondary = setTimeout(async () => {
      if (isIntentionalDisconnectSecondary) return;
      
      // Handle Simulation Reconnect
      if (portPath === 'SIMULATION' || portPath === 'SIMULATION_MODE') {
          console.log('Reconnecting to Secondary Simulation...');
          startSimulationSecondary();
          return;
      }

      console.log(`Attempting to reconnect secondary to ${portPath}...`);

      // Check if device is present to avoid rapid error loops if desired, 
      // OR just try connecting. SerialPort throws if not found.
      try {
        const ports = await SerialPort.list();
        const exists = ports.some(p => p.path === portPath);
        if (exists) {
          connectToSecondaryPort(portPath, baudRate);
        } else {
          console.log(`Secondary Port ${portPath} not found yet. Retrying...`);
          attemptReconnect(portPath, baudRate, true);
        }
      } catch (e) {
        console.log('Reconnect check failed', e);
        attemptReconnect(portPath, baudRate, true);
      }
    }, retryDelay);
  } else {
    if (reconnectTimer) clearTimeout(reconnectTimer);
    reconnectTimer = setTimeout(async () => {
      if (isIntentionalDisconnect) return;

      // Handle Simulation Reconnect
      if (portPath === 'SIMULATION' || portPath === 'SIMULATION_MODE') {
        console.log('Reconnecting to Primary Simulation...');
        startSimulation();
        return;
      }

      console.log(`Attempting to reconnect primary to ${portPath}...`);

      try {
        const ports = await SerialPort.list();
        const exists = ports.some(p => p.path === portPath);
        if (exists) {
          connectToPort(portPath, baudRate);
        } else {
          console.log(`Primary Port ${portPath} not found yet. Retrying...`);
          attemptReconnect(portPath, baudRate, false);
        }
      } catch (e) {
        console.log('Reconnect check failed', e);
        attemptReconnect(portPath, baudRate, false);
      }
    }, retryDelay);
  }
}

function handleSerialData(data) {
  console.log('Received Data:', data);
  const trimmed = data.trim();
  if (!trimmed) return;

  // Expected format: "0.2,150,1,2" (Rpm,voltaje,velocidad,4x4,palanca)
  const parts = trimmed.split(',');
  if (parts.length >= 2) {
    const rpm = parseFloat(parts[0]);
    const voltaje = parseFloat(parts[1]);
    const velocidad = parseFloat(parts[2]);
    const road = parseFloat(parts[3]);
    const cambios = parseFloat(parts[4]);
    const fuel_lvl = parseFloat(parts[5]);
    const oil_press = parseFloat(parts[6]);
    const fan1 = parseFloat(parts[7]);
    const fan2 = parseFloat(parts[8]);
    const fuel_press = parseFloat(parts[9]);
    if (!isNaN(rpm) && !isNaN(voltaje) && !isNaN(velocidad)) {
      if (mainWindow) mainWindow.webContents.send('serial:data', { rpm, voltaje, velocidad, road, cambios, fuel_lvl, oil_press, fan1, fan2, fuel_press, timestamp: Date.now(), valid: true, rawData: data });
    }
  } else {
    if (mainWindow) mainWindow.webContents.send('serial:data', { timestamp: Date.now(), valid: false, rawData: data });
  }
}

// --- OBD Logic ---
// PIDs configuration based on python script logic
const PIDS = {
  '010C': { name: 'rpm', parse: (bytes) => (bytes[0] * 256 + bytes[1]) / 4 },
  '010D': { name: 'speed', parse: (bytes) => bytes[0] },
  '0105': { name: 'coolant', parse: (bytes) => bytes[0] - 40 },
  '0104': { name: 'load', parse: (bytes) => bytes[0] * 100 / 255 },
  '010F': { name: 'iat', parse: (bytes) => bytes[0] - 40 },
  '0110': { name: 'maf', parse: (bytes) => (bytes[0] * 256 + bytes[1]) / 100 },
  '0111': { name: 'tps', parse: (bytes) => bytes[0] * 100 / 255 },
  '010B': { name: 'map', parse: (bytes) => bytes[0] },
  '010E': { name: 'timing', parse: (bytes) => (bytes[0] - 128) / 2 },
  '0103': { name: 'fuel_status', parse: (bytes) => bytes[0] + bytes[1] * 256 }, // Simplified
};

// Queue state
let obdQueue = [];
let obdData = {};
let isObdInitialized = false;
let obdPollInterval = null;
let currentCmd = null;

function resetObdState() {
  obdQueue = [];
  obdData = {};
  isObdInitialized = false;
  currentCmd = null;
  // if (obdPollInterval) clearInterval(obdPollInterval); // Not using interval, self-clocked by responses
}

function processObdResponse(data) {
  // Simple ELM327 response parser
  // format: "41 0C 0F A0" or "41 0C \r 0F A0 >"

  const clean = data.replace(/>/g, '').trim();
  if (!clean) return;

  const lines = clean.split('\r');
  lines.forEach(line => {
    line = line.trim();
    // Check for 41 xx ...
    if (line.startsWith('41')) {
      const parts = line.split(' ');
      // 41 PID A B ...
      if (parts.length >= 3) {
        const pid = '01' + parts[1];
        // Safe hex parsing
        const bytes = [];
        for (let i = 2; i < parts.length; i++) {
          bytes.push(parseInt(parts[i], 16));
        }

        if (PIDS[pid]) {
          const val = PIDS[pid].parse(bytes);
          obdData[PIDS[pid].name] = parseFloat(val.toFixed(2));
        }
      }
    }
  });

  if (data.includes('>')) {
    sendNextObdCommand();
  }
}

function sendNextObdCommand() {
  if (!secondaryPort || !secondaryPort.isOpen) return;

  // Fill queue if empty and initialized
  if (obdQueue.length === 0) {
    if (isObdInitialized) {
      obdQueue = Object.keys(PIDS).map(k => k.substring(2)); // ['0C', '0D'...]
      // Emit data at end of cycle
      if (mainWindow) mainWindow.webContents.send('serial:data-secondary', obdData);
    } else {
      return;
    }
  }

  if (obdQueue.length > 0) {
    let cmd = obdQueue.shift();
    currentCmd = cmd;
    // Append Mode 01 if it's a 2-char PID
    if (cmd.length === 2 && /^[0-9A-F]+$/i.test(cmd)) {
      cmd = '01' + cmd;
    }
    secondaryPort.write(cmd + '\r');
  }
}

function startObdPolling() {
  resetObdState();
  // Init Sequence
  obdQueue = ['AT Z', 'AT E0', 'AT L0', 'AT H0', 'AT SP 0'];

  // Kickoff
  setTimeout(() => {
    sendNextObdCommand();
  }, 1000);
}

function handleSecondarySerialData(data) {
  // Check Init Phase
  if (!isObdInitialized) {
    // If we see '>', assume command finished
    if (data.includes('>')) {
      // If queue is empty, we finished init
      if (obdQueue.length === 0 && currentCmd) {
        console.log('OBD Intialized');
        isObdInitialized = true;
        sendNextObdCommand(); // Will refill queue with PIDs
        return;
      }
      sendNextObdCommand();
    }
  } else {
    // Polling Phase
    processObdResponse(data);
  }
}

function connectToSecondaryPort(portPath, baudRate = 115200) {
  isIntentionalDisconnectSecondary = false;
  if (reconnectTimerSecondary) {
    clearTimeout(reconnectTimerSecondary);
    reconnectTimerSecondary = null;
  }

  if (secondaryPort && secondaryPort.isOpen) {
    secondaryPort.close();
  }

  try {
    secondaryPort = new SerialPort({ path: portPath, baudRate: parseInt(baudRate) });
    // Use default parser but we append > manually if needed in handler?
    // Actually, ELM327 outputs > as a prompt, usually without newline after it.
    // Node serialport ReadlineParser might wait forever if no \n comes after >.
    // Let's use '>' as delimiter.

    const parserSec = secondaryPort.pipe(new ReadlineParser({ delimiter: '>' }));

    secondaryPort.on('open', () => {
      console.log(`Connected to Secondary ${portPath}`);
      if (mainWindow) mainWindow.webContents.send('serial:status-secondary', { connected: true, port: portPath });
      saveSecondaryPortInfo(portPath, baudRate);

      // Start OBD
      startObdPolling();
    });

    secondaryPort.on('error', (err) => {
      console.error('Secondary Serial Port Error:', err);
      if (!isIntentionalDisconnectSecondary) {
        if (mainWindow) mainWindow.webContents.send('serial:status-secondary', { connected: false, error: err.message, reconnecting: true });

        if (!secondaryPort.isOpen) {
          attemptReconnect(portPath, baudRate, true);
        }
      } else {
        if (mainWindow) mainWindow.webContents.send('serial:status-secondary', { connected: false, error: err.message });
      }
    });

    secondaryPort.on('close', () => {
      console.log('Secondary Port closed');
      resetObdState();

      if (!isIntentionalDisconnectSecondary) {
        console.log('Secondary Port disconnected unexpectedly. Attempting reconnect...');
        if (mainWindow) mainWindow.webContents.send('serial:status-secondary', { connected: false, reconnecting: true });
        attemptReconnect(portPath, baudRate, true);
      } else {
        if (mainWindow) mainWindow.webContents.send('serial:status-secondary', { connected: false });
      }
    });

    parserSec.on('data', (data) => {
      // Re-append > for parsing logic
      handleSecondarySerialData(data + '>');
    });

    return { success: true };
  } catch (err) {
    return { success: false, error: err.message };
  }
}

async function savePortInfo(portPath, baudRate) {
  const ports = await SerialPort.list();
  const portDetails = ports.find(p => p.path === portPath);
  if (portDetails && portDetails.vendorId && portDetails.productId) {
    store.set('lastDevice', {
      vendorId: portDetails.vendorId,
      productId: portDetails.productId,
      baudRate: baudRate
    });
    console.log('Saved Persistent Device:', portDetails.vendorId, portDetails.productId);
  }
}

async function saveSecondaryPortInfo(portPath, baudRate) {
  const ports = await SerialPort.list();
  const portDetails = ports.find(p => p.path === portPath);
  if (portDetails && portDetails.vendorId && portDetails.productId) {
    store.set('lastDeviceSecondary', {
      vendorId: portDetails.vendorId,
      productId: portDetails.productId,
      baudRate: baudRate
    });
    console.log('Saved Persistent Secondary Device:', portDetails.vendorId, portDetails.productId);
  }
}

async function attemptAutoConnect() {
  const lastDevice = store.get('lastDevice');
  if (lastDevice && lastDevice.vendorId && lastDevice.productId) {
    console.log('Looking for known device...', lastDevice);
    const ports = await SerialPort.list();
    const found = ports.find(p => p.vendorId === lastDevice.vendorId && p.productId === lastDevice.productId);
    if (found) {
      console.log('Found known device on', found.path);
      connectToPort(found.path, lastDevice.baudRate || 115200);
    } else {
      console.log('Known device not found.');
    }
  }

  const lastDeviceSecondary = store.get('lastDeviceSecondary');
  if (lastDeviceSecondary && lastDeviceSecondary.vendorId && lastDeviceSecondary.productId) {
    console.log('Looking for known secondary device...', lastDeviceSecondary);
    const ports = await SerialPort.list();
    const found = ports.find(p => p.vendorId === lastDeviceSecondary.vendorId && p.productId === lastDeviceSecondary.productId);
    if (found) {
      console.log('Found known secondary device on', found.path);
      connectToSecondaryPort(found.path, lastDeviceSecondary.baudRate || 115200);
    } else {
      console.log('Known secondary device not found.');
    }
  }
}

// --- Simulation Logic ---

function startSimulation() {
  if (currentPort && currentPort.isOpen) {
    currentPort.close();
  }
  isSimulationMode = true;
  mainWindow.webContents.send('serial:status', { connected: true, port: 'SIMULATION_MODE' });

  let rpm = 0; //min = 0 , max = 6000
  let velocidad = 0; //min = 0 , max = 300
  let road = 0; // 0 = 4x4, 1 = 2x2
  let cambios = 0; // 0 = P, 1 = R, 2 = N, 3 = D, 4 = 2, 5 = 1
  let voltaje = 0; //min = 0 , max = 300
  let fuel_lvl = 0; //min = 0 , max = 100
  let oil_press = 0; //min = 0 , max = 100
  let fan1 = 0; //min = 0 , max = 100
  let fan2 = 0; //min = 0 , max = 100
  let fuel_press = 0; //min = 0 , max = 100
  simulationInterval = setInterval(() => {
    // Random walk
    rpm = Math.round(Math.random() * 6000);
    velocidad = Math.round(Math.random() * 120);
    road = Math.round(Math.random() * 1);
    cambios = Math.round(Math.random() * 5);
    voltaje = Math.round(Math.random() * 15);
    fuel_lvl = Math.round(Math.random() * 100);
    oil_press = Math.round(Math.random() * 100);
    fan1 = Math.round(Math.random() * 1500);
    fan2 = Math.round(Math.random() * 1500);
    fuel_press = Math.round(Math.random() * 100);
    mainWindow.webContents.send('serial:data', {
      rpm: parseFloat(rpm.toFixed(1)),
      velocidad: parseFloat(velocidad.toFixed(1)),
      voltaje: parseFloat(voltaje.toFixed(1)),
      road: parseFloat(road.toFixed(1)),
      cambios: parseFloat(cambios.toFixed(1)),
      fuel_lvl: parseFloat(fuel_lvl.toFixed(1)),
      oil_press: parseFloat(oil_press.toFixed(1)),
      fan1: parseFloat(fan1.toFixed(1)),
      fan2: parseFloat(fan2.toFixed(1)),
      fuel_press: parseFloat(fuel_press.toFixed(1)),
      timestamp: Date.now(),
      valid: true
    });
  }, 1000); // 1Hz update
}

function stopSimulation() {
  isSimulationMode = false;
  if (simulationInterval) {
    clearInterval(simulationInterval);
    simulationInterval = null;
  }
}

function startSimulationSecondary() {
  // Stop any pending reconnect attempts
  if (reconnectTimerSecondary) {
    clearTimeout(reconnectTimerSecondary);
    reconnectTimerSecondary = null;
  }
  // Prevent auto-reconnect logic from triggering if we were in a loop
  isIntentionalDisconnectSecondary = true; 

  if (secondaryPort && secondaryPort.isOpen) {
    secondaryPort.close();
  }
  
  // Now we are intentionally "connecting" to simulation, so we can reset intention if needed, 
  // but simpler to just ensure clean state.
  isSimulationModeSecondary = true;
  if (mainWindow) mainWindow.webContents.send('serial:status-secondary', { connected: true, port: 'SIMULATION_MODE', reconnecting: false });

  // PIDS: rpm, speed, coolant, load, iat, maf, tps, map, timing
  let rpm = 0;
  let speed = 0;
  
  simulationIntervalSecondary = setInterval(() => {
    rpm = Math.round(Math.random() * 6000);
    speed = Math.round(Math.random() * 120);
    
    // Simulate OBD Data map
    // Note: Serial format for OBD usually comes as responses.
    // However, our `serial:data-secondary` event expects the processed map { name: value }
    // because `startObdPolling` emits `obdData` map.
    const simData = {
      rpm: rpm,
      speed: speed,
      coolant: Math.round(50 + Math.random() * 50),
      load: parseFloat((Math.random() * 100).toFixed(1)),
      iat: Math.round(20 + Math.random() * 30),
      maf: parseFloat((Math.random() * 50).toFixed(2)),
      tps: parseFloat((Math.random() * 100).toFixed(1)),
      map: Math.round(Math.random() * 200),
      timing: Math.round(Math.random() * 40 - 10),
      fuel_status: 3
    };

    if (mainWindow) mainWindow.webContents.send('serial:data-secondary', simData);
  }, 1000);
}

function stopSimulationSecondary() {
  isSimulationModeSecondary = false;
  if (simulationIntervalSecondary) {
    clearInterval(simulationIntervalSecondary);
    simulationIntervalSecondary = null;
  }
}

// --- IPC Handlers ---

ipcMain.handle('serial:list', async () => {
  return await listPorts();
});

ipcMain.handle('get:ip', async () => {
  return await getIP();
});

ipcMain.handle('app:get-version', () => {
  return app.getVersion();
});

ipcMain.handle('serial:get-status', () => {
  return {
    connected: (currentPort && currentPort.isOpen) || isSimulationMode,
    reconnecting: !!reconnectTimer,
    port: isSimulationMode ? 'SIMULATION_MODE' : (currentPort ? currentPort.path : null),
    baudRate: currentPort ? currentPort.baudRate : null
  };
});

ipcMain.handle('serial:connect', async (event, { path: portPath, baudRate }) => {
  if (portPath === 'SIMULATION') {
    startSimulation();
    return { success: true };
  }

  // If already connected to this port, just return success
  if (currentPort && currentPort.isOpen && currentPort.path === portPath) {
    console.log('Already connected to', portPath);
    return { success: true };
  }

  isIntentionalDisconnect = true; // Mark as intentional switch if we are changing
  if (currentPort && currentPort.isOpen) {
    currentPort.close();
  }

  // Wait a tick for close to process? Not strictly needed if sync, but good practice.
  // Then connect new.

  return connectToPort(portPath, baudRate);
});

ipcMain.handle('serial:write', async (event, data) => {
  if (isSimulationMode) {
    console.log('SIMULATION WRITE:', data);
    return { success: true };
  }
  if (!currentPort || !currentPort.isOpen) {
    return { success: false, error: 'Port not open' };
  }
  try {
    currentPort.write(data);
    return { success: true };
  } catch (err) {
    console.error('Serial Write Error:', err);
    return { success: false, error: err.message };
  }
});

ipcMain.handle('serial:disconnect', async () => {
  isIntentionalDisconnect = true;
  if (reconnectTimer) clearTimeout(reconnectTimer);

  if (mainWindow) mainWindow.webContents.send('serial:status', { connected: false });

  if (isSimulationMode) {
    stopSimulation();
    return { success: true };
  }
  if (currentPort && currentPort.isOpen) {
    currentPort.close();
  }
  // Always success if we stopped the intention to connect
  return { success: true };
});

ipcMain.handle('serial:get-status-secondary', () => {
  return {
    connected: (secondaryPort && secondaryPort.isOpen) || isSimulationModeSecondary,
    reconnecting: !!reconnectTimerSecondary,
    port: isSimulationModeSecondary ? 'SIMULATION_MODE' : (secondaryPort ? secondaryPort.path : null),
    baudRate: secondaryPort ? secondaryPort.baudRate : null
  };
});

ipcMain.handle('serial:connect-secondary', async (event, { path: portPath, baudRate }) => {
  if (portPath === 'SIMULATION') {
    startSimulationSecondary();
    return { success: true };
  }

  if (secondaryPort && secondaryPort.isOpen && secondaryPort.path === portPath) {
    return { success: true };
  }
  isIntentionalDisconnectSecondary = true; // Switch
  if (secondaryPort && secondaryPort.isOpen) {
    secondaryPort.close();
  }
  return connectToSecondaryPort(portPath, baudRate);
});

ipcMain.handle('serial:disconnect-secondary', async () => {
  isIntentionalDisconnectSecondary = true;
  if (reconnectTimerSecondary) clearTimeout(reconnectTimerSecondary);

  if (mainWindow) mainWindow.webContents.send('serial:status-secondary', { connected: false });

  if (isSimulationModeSecondary) {
    stopSimulationSecondary();
    return { success: true };
  }

  if (secondaryPort && secondaryPort.isOpen) {
    secondaryPort.close();
  }
  return { success: true };
});

// --- WiFi Management Handlers ---
const { exec } = require('child_process');
const util = require('util');
const execPromise = util.promisify(exec);

// Scan for available WiFi networks
ipcMain.handle('wifi:scan', async () => {
  try {
    const { stdout } = await execPromise('nmcli -t -f SSID,SIGNAL,SECURITY device wifi list');
    
    // Parse the output
    const lines = stdout.trim().split('\n');
    const networks = lines.map(line => {
      const parts = line.split(':');
      if (parts.length >= 2) {
        return {
          ssid: parts[0] || 'Hidden Network',
          signal: parseInt(parts[1]) || 0,
          security: parts[2] || 'Open'
        };
      }
      return null;
    }).filter(n => n !== null && n.ssid !== '');
    
    // Remove duplicates and sort by signal strength
    const uniqueNetworks = Array.from(
      new Map(networks.map(n => [n.ssid, n])).values()
    ).sort((a, b) => b.signal - a.signal);
    
    return { success: true, networks: uniqueNetworks };
  } catch (err) {
    console.error('WiFi scan error:', err);
    return { success: false, error: err.message, networks: [] };
  }
});

// Connect to a WiFi network
ipcMain.handle('wifi:connect', async (event, { ssid, password }) => {
  try {
    let cmd;
    if (password) {
      // Connect with password (secured network)
      cmd = `nmcli device wifi connect "${ssid}" password "${password}"`;
    } else {
      // Connect without password (open network)
      cmd = `nmcli device wifi connect "${ssid}"`;
    }
    
    // Add timeout to prevent hanging (30 seconds)
    try {
        const { stdout } = await execPromise(cmd, { timeout: 30000 });
        console.log('WiFi connect output:', stdout);
        return { success: true, message: 'Connected successfully' };
    } catch(initialError) {
        // Check for permission error and retry with sudo
        if (initialError.message.includes('Not authorized')) {
             console.log('Permission error detected, retrying with sudo...');
             const sudoCmd = `sudo ${cmd}`;
             const { stdout } = await execPromise(sudoCmd, { timeout: 30000 });
             console.log('WiFi connect output (sudo):', stdout);
             return { success: true, message: 'Connected successfully' };
        }
        throw initialError; // Re-throw if it's not a permission error or if sudo also fails
    }

  } catch (err) {
    console.error('WiFi connect error:', err);
    // Parse common error messages
    let errorMsg = err.message;
    if (errorMsg.includes('Secrets were required')) {
      errorMsg = 'Contraseña incorrecta';
    } else if (errorMsg.includes('No network with SSID')) {
      errorMsg = 'Red no encontrada';
    } else if (err.killed && err.signal === 'SIGTERM') {
      errorMsg = 'Tiempo de espera agotado al intentar conectar';
    } else if (errorMsg.includes('Not authorized')) {
       errorMsg = 'Error de permisos: No autorizado para controlar la red (intente ejecutar como root/sudo)';
    }
    return { success: false, error: errorMsg };
  }
});

// Get current WiFi status
ipcMain.handle('wifi:status', async () => {
  try {
    const { stdout } = await execPromise('nmcli -t -f ACTIVE,SSID,SIGNAL device wifi list');
    
    // Find the active connection (marked with 'yes' in ACTIVE field)
    const lines = stdout.trim().split('\n');
    const activeLine = lines.find(line => line.startsWith('yes:'));
    
    if (activeLine) {
      const parts = activeLine.split(':');
      return {
        connected: true,
        ssid: parts[1] || '',
        signal: parseInt(parts[2]) || 0
      };
    }
    
    return { connected: false };
  } catch (err) {
    console.error('WiFi status error:', err);
    return { connected: false, error: err.message };
  }
});

// Disconnect from WiFi
ipcMain.handle('wifi:disconnect', async () => {
  try {
    // Get the WiFi device name (usually wlan0 or similar)
    const { stdout: devList } = await execPromise('nmcli -t -f DEVICE,TYPE device');
    const wifiDevice = devList.split('\n').find(line => line.includes(':wifi'));
    
    if (wifiDevice) {
      const deviceName = wifiDevice.split(':')[0];
      await execPromise(`nmcli device disconnect ${deviceName}`);
      return { success: true, message: 'Disconnected successfully' };
    }
    
    return { success: false, error: 'No WiFi device found' };
  } catch (err) {
    console.error('WiFi disconnect error:', err);
    return { success: false, error: err.message };
  }
});

// --- On-Screen Keyboard Handlers ---
let keyboardProcess = null;

// Show the on-screen keyboard
ipcMain.handle('keyboard:show', async () => {
  try {
    console.log('Attempting to show keyboard...');
    
    // Only launch if not already running
    if (!keyboardProcess) {
      console.log('Launching matchbox-keyboard...');
      keyboardProcess = exec('matchbox-keyboard', (error, stdout, stderr) => {
        if (error) {
          console.error('Keyboard process error:', error);
          console.error('stderr:', stderr);
          keyboardProcess = null;
          return;
        }
        if (stdout) console.log('Keyboard stdout:', stdout);
        if (stderr) console.log('Keyboard stderr:', stderr);
      });
      
      // Clean up reference when process exits
      keyboardProcess.on('exit', (code, signal) => {
        console.log(`Keyboard process exited with code ${code} and signal ${signal}`);
        keyboardProcess = null;
      });
      
      keyboardProcess.on('error', (err) => {
        console.error('Failed to start keyboard process:', err);
        keyboardProcess = null;
      });
      
      console.log('Keyboard process started');
    } else {
      console.log('Keyboard already running');
    }
    
    return { success: true };
  } catch (err) {
    console.error('Error showing keyboard:', err);
    return { success: false, error: err.message };
  }
});

// Hide the on-screen keyboard
ipcMain.handle('keyboard:hide', async () => {
  try {
    console.log('Attempting to hide keyboard...');
    
    if (keyboardProcess) {
      console.log('Killing keyboard process...');
      keyboardProcess.kill();
      keyboardProcess = null;
    }
    
    // Also kill any running matchbox-keyboard processes
    exec('pkill -f matchbox-keyboard', (error, stdout, stderr) => {
      if (error) {
        // pkill returns error code 1 if no process found, which is not really an error
        if (error.code === 1) {
          console.log('No keyboard process to kill');
        } else {
          console.error('Error killing keyboard:', error);
        }
      } else {
        console.log('Keyboard process killed successfully');
      }
    });
    
    return { success: true };
  } catch (err) {
    console.error('Error hiding keyboard:', err);
    return { success: false, error: err.message };
  }
});




app.on('ready', createWindow);

app.on('window-all-closed', function () {
  if (process.platform !== 'darwin') app.quit();
});

app.on('activate', function () {
  if (mainWindow === null) createWindow();
});
